<?php
/**
 * PropertyModifierReader.php 2020-3-24
 * Gambio GmbH
 * http://www.gambio.de
 * Copyright (c) 2020 Gambio GmbH
 * Released under the GNU General Public License (Version 2)
 * [http://www.gnu.org/licenses/gpl-2.0.html]
 */
declare(strict_types=1);

namespace Gambio\Shop\Properties\ProductModifiers\Database\Readers;

use CI_DB_query_builder;
use Gambio\Shop\ProductModifiers\Database\Core\DTO\Modifiers\ModifierDTOBuilderInterface;
use Gambio\Shop\ProductModifiers\Database\Core\DTO\Modifiers\ModifierDTOCollection;
use Gambio\Shop\ProductModifiers\Database\Core\DTO\Modifiers\ModifierDTOCollectionInterface;
use Gambio\Shop\ProductModifiers\Database\Core\Readers\Interfaces\ModifierReaderCompositeInterface;
use Gambio\Shop\Properties\ProductModifiers\Database\ValueObjects\PropertyGroupIdentifier;
use Gambio\Shop\Properties\ProductModifiers\Database\ValueObjects\PropertyModifierIdentifier;
use Gambio\Shop\Product\ValueObjects\ProductId;
use IdType;

/**
 * Class PropertyModifierReader
 * @package Gambio\Shop\Properties\ProductModifiers\Database\Readers
 */
class PropertyModifierReader implements ModifierReaderCompositeInterface
{
    
    /**
     * @var ModifierDTOBuilderInterface
     */
    protected $builder;
    /**
     * @var CI_DB_query_builder
     */
    protected $queryBuilder;
    
    
    /**
     * PropertyModifierReader constructor.
     *
     * @param CI_DB_query_builder        $queryBuilder
     * @param ModifierDTOBuilderInterface $builder
     */
    public function __construct(CI_DB_query_builder $queryBuilder, ModifierDTOBuilderInterface $builder)
    {
        $this->queryBuilder = $queryBuilder;
        $this->builder      = $builder;
    }
    
    
    /**
     * @inheritDoc
     */
    public function getModifierByProduct(ProductId $id, IdType $languageId): ModifierDTOCollectionInterface
    {
        $result = new ModifierDTOCollection();
        
        $sql = "SELECT p.properties_id, p.display_type, pv.display_image, pvd.values_name, pvd.properties_values_id, pv.`value_price`
                    FROM products_properties_combis ppc
                        INNER JOIN products_properties_combis_values ppcv
                            ON ppc.products_properties_combis_id = ppcv.products_properties_combis_id
                        INNER JOIN properties_values_description pvd
                            ON pvd.properties_values_id = ppcv.properties_values_id AND pvd.language_id = {$languageId->asInt()}
                        INNER JOIN properties_values pv
                            ON pv.properties_values_id = ppcv.properties_values_id
                        INNER JOIN properties p
                            ON p.properties_id = pv.properties_id
                    WHERE ppc.products_id = {$id->value()}
                    GROUP BY p.properties_id, pvd.values_name, pvd.properties_values_id
                    ORDER BY ppc.sort_order";
        
        $data = $this->queryBuilder->query($sql)->result_array();
        
        foreach ($data as $item) {
    
            [$pricePrefix, $priceValueFloat] = $this->pricePrefixAndValueFromFloat((float)$item['value_price']);
            
            $result->addModifier($this->builder->withId(new PropertyModifierIdentifier($item['properties_values_id']))
                                     ->withGroupId(new PropertyGroupIdentifier((int)$item['properties_id']))
                                     ->withType($item['display_type'])
                                     ->withName($item['values_name'])
                                     ->withPrice($priceValueFloat)
                                     ->withPricePrefix($pricePrefix)
                                     ->withImage($item['display_image'] ? 'product_images/property_images/' . $item['display_image'] : '')
                                     ->withSource('property')
                                     ->build());
        }
        
        return $result;
    }
    
    
    /**
     * @param float $priceValueFloat
     *
     * @return mixed[]
     */
    protected function pricePrefixAndValueFromFloat(float $priceValueFloat): array
    {
        $pricePrefix     = $priceValueFloat >= 0 ? '+ ' : '- ';
        $priceValueFloat = $pricePrefix === '- ' ? $priceValueFloat * -1 : $priceValueFloat;
        
        return [$pricePrefix, $priceValueFloat];
    }
}